<?php
namespace App\Services;
use App\Core\DB;

class RouterManager {
  public static function listRouters(){
    $stmt = DB::pdo()->query('SELECT * FROM routers');
    return $stmt->fetchAll(\PDO::FETCH_ASSOC);
  }

  public static function addRouter($data){
    $stmt = DB::pdo()->prepare('INSERT INTO routers (name, ip, api_port, api_user, api_pass, public_key, allowed_ips, owner_user_id, notes) VALUES (?,?,?,?,?,?,?,?,?)');
    return $stmt->execute([$data['name'],$data['ip'],$data['api_port'],$data['api_user'],$data['api_pass'],$data['public_key'],$data['allowed_ips'],$data['owner_user_id'],$data['notes']]);
  }

  public static function createHotspotUserOnRouter($router, $username, $password, $profile=null){
    try {
      if(class_exists('\\RouterOS\\Client')){
        $client = new \RouterOS\Client(new \RouterOS\Config(['host'=>$router['ip'],'user'=>$router['api_user'],'pass'=>$router['api_pass'],'port'=>$router['api_port']]));
        $request = new \RouterOS\Request('/ip/hotspot/user/add');
        $request->setArgument('name', $username);
        $request->setArgument('password', $password);
        if($profile) $request->setArgument('profile', $profile);
        $client->send($request);
        \App\Services\Audit::log(null, 'provisioned_hotspot_user', 'hotspot_user', null, ['router'=>$router['name'],'username'=>$username]);
        return true;
      } else {
        throw new \Exception('RouterOS client not available');
      }
    } catch(\Exception $e){
      try {
        $cmds = ["/ip hotspot user add name={$username} password={$password}" . ($profile ? " profile={$profile}" : "")];
        return self::provisionViaSSH($router, $cmds);
      } catch(\Exception $ex){
        DB::pdo()->prepare('INSERT INTO logs (level,source,message) VALUES (?,?,?)')->execute(['error','RouterManager',$ex->getMessage()]);
        return false;
      }
    }
  }

  public static function createPPPoEUserOnRouter($router, $username, $password, $profile=null){
    try {
      if(class_exists('\\RouterOS\\Client')){
        $client = new \RouterOS\Client(new \RouterOS\Config(['host'=>$router['ip'],'user'=>$router['api_user'],'pass'=>$router['api_pass'],'port'=>$router['api_port']]));
        $req = new \RouterOS\Request('/ppp/secret/add');
        $req->setArgument('name', $username);
        $req->setArgument('password', $password);
        $req->setArgument('service','pppoe');
        if($profile) $req->setArgument('profile',$profile);
        $client->send($req);
        \App\Services\Audit::log(null, 'provisioned_pppoe_user', 'pppoe_user', null, ['router'=>$router['name'],'username'=>$username]);
        return true;
      } else {
        throw new \Exception('RouterOS client not available');
      }
    } catch(\Exception $e){
      try {
        $cmds = ["/ppp secret add name={$username} password={$password} service=pppoe" . ($profile ? " profile={$profile}" : "")];
        return self::provisionViaSSH($router, $cmds);
      } catch(\Exception $ex){
        DB::pdo()->prepare('INSERT INTO logs (level,source,message) VALUES (?,?,?)')->execute(['error','RouterManager',$ex->getMessage()]);
        return false;
      }
    }
  }

  private static function provisionViaSSH($router, $commands){
    $config = require __DIR__ . '/../../config.php';
    $sshUser = $config['routeros_fallback']['ssh_user'];
    $sshKey = $config['routeros_fallback']['ssh_key_path'];
    $port = $config['routeros_fallback']['ssh_port'] ?? 22;

    if(!file_exists($sshKey)) throw new \Exception('SSH key not found on provisioning host');

    $ssh = new \phpseclib3\Net\SSH2($router['ip'], $port);
    $priv = \phpseclib3\Crypt\PublicKeyLoader::loadPrivateKey(file_get_contents($sshKey));
    if(!$ssh->login($sshUser, $priv)) throw new \Exception('SSH login failed');
    foreach($commands as $cmd){
      $out = $ssh->exec($cmd);
    }
    \App\Services\Audit::log(null, 'provisioned_via_ssh', 'router', $router['id'], ['commands'=>$commands]);
    return true;
  }
}
